//////////////////////////////////////////////////////////////////////////////// 
// Auto-Lock CD-ROM Door Plugin
// 
// Copyright  2006  Sebastian Pipping <webmaster@hartwork.org>
// 
// -->  http://www.hartwork.org
// 
// This source code is released under the GNU General Public License (GPL).
// See GPL.txt for details. Any non-GPL usage is strictly forbidden.
////////////////////////////////////////////////////////////////////////////////


#define WIN32_LEAN_AND_MEAN
#include <windows.h> 
#include "Winamp/Gen.h"
#include "Winamp/wa_ipc.h" 
#include "resource.h"
#include <Winioctl.h>



#define PLUGIN_TITLE    "Auto-Lock CD-ROM Door Plugin"
#define PLUGIN_VERSION  "1.1"



int init();
void config(); 
void quit();



WNDPROC WndprocMainBackup = NULL;
LRESULT CALLBACK WndprocMain( HWND hwnd, UINT message, WPARAM wp, LPARAM lp );

char szLastDevice[] = "\\\\.\\?:";
char * const szDrive = szLastDevice + 4;
bool bLocked = false;
bool bActive;

char * szWinampIni;



winampGeneralPurposePlugin plugin = {
	GPPHDR_VER,
	PLUGIN_TITLE " " PLUGIN_VERSION,
	init,
	config, 
	quit,
	NULL,
	NULL
}; 



void lock( const bool bPositive )
{
	// How To Ejecting Removable Media in Windows NT/Windows 2000/Windows XP
	// http://support.microsoft.com/default.aspx?scid=kb;en-us;165721

	// IOCTL_STORAGE_MEDIA_REMOVAL versus IOCTL_STORAGE_EJECTION_CONTROL
	// http://msdn.microsoft.com/library/default.asp?url=/library/en-us/Storage_r/hh/Storage_r/k307_fe4c03f1-b835-4c04-bb7b-4a8bd982dcc7.xml.asp

	// Anything to do?
	if( bPositive == bLocked )
	{
#ifdef _DEBUG
		if( bLocked )
			OutputDebugString( "AUTOLOCK - Already locked" );
		else
			OutputDebugString( "AUTOLOCK - Already unlocked" );
#endif

		return;
	}
	
	// Locking allowed?
	if( bPositive && !bActive )
	{
#ifdef _DEBUG
		OutputDebugString( "AUTOLOCK - Not active" );
		return;
#endif
	}

	const HANDLE hDevice = CreateFile(
		szLastDevice,
		GENERIC_READ,
		FILE_SHARE_READ | FILE_SHARE_WRITE,
		NULL,
		OPEN_EXISTING,
		0,
		NULL
	);
	
	if( hDevice == INVALID_HANDLE_VALUE ) return;

	DWORD dwBytesReturned = 0;

	PREVENT_MEDIA_REMOVAL pmr;
	pmr.PreventMediaRemoval = bPositive ? TRUE : FALSE;

	if( DeviceIoControl(
		hDevice,
		IOCTL_STORAGE_MEDIA_REMOVAL,
		&pmr,
		sizeof( PREVENT_MEDIA_REMOVAL ),
		NULL,
		0,
		&dwBytesReturned,
		NULL ) )
	{
#ifdef _DEBUG		
		if( bPositive ) 
			OutputDebugString( "AUTOLOCK - Locked" );
		else
			OutputDebugString( "AUTOLOCK - Unlocked" );
#endif

		bLocked = bPositive;
	}

	CloseHandle( hDevice );
}



LRESULT CALLBACK WndprocMain( HWND hwnd, UINT message, WPARAM wp, LPARAM lp )
{
	switch( message )
	{
	case WM_WA_IPC:
		if( ( lp == IPC_CB_MISC ) && ( wp == IPC_CB_MISC_STATUS ) )
		{
			switch( SendMessage( hwnd, WM_WA_IPC, 0, IPC_ISPLAYING ) )
			{
			// Start
			case 1:
				if( SendMessage( hwnd, WM_WA_IPC, 0, IPC_GETOUTPUTTIME ) >= 0 )
				{
#ifdef _DEBUG					
					OutputDebugString( "AUTOLOCK - 1 - Play" );
#endif

					const char * const szFile = ( char * )SendMessage(
						hwnd,
						WM_WA_IPC,
						SendMessage( hwnd, WM_WA_IPC, 0, IPC_GETLISTPOS ),
						IPC_GETPLAYLISTFILE
					);
					
					if( szFile )
					{
#ifdef _DEBUG
						OutputDebugString( "AUTOLOCK - 1.1 - Got filename" );
#endif

						// "cda://?,x"
						if( ( strlen( szFile ) >= 9 )
							&& !strncmp( szFile, "cda://", 6 )
							&& ( szFile[ 7 ] == ',' ) )
						{
#ifdef _DEBUG							
							OutputDebugString( "AUTOLOCK - 1.1.1 - CDA track" );
#endif

							// New drive or same again?
							if( szFile[ 6 ] != *szDrive )
							{
#ifdef _DEBUG
								OutputDebugString( "AUTOLOCK - 1.1.1.1 - New drive" );
#endif

								// Unlock drive locked last time
								lock( false );

								// Lock new drive
								*szDrive = szFile[ 6 ];
								lock( true );
							}
#ifdef _DEBUG
							else
							{
								OutputDebugString( "AUTOLOCK - 1.1.1.2 - Same drive" );
							}
#endif
						}
						else
						{
#ifdef _DEBUG
							OutputDebugString( "AUTOLOCK - 1.1.2 - Non-CDA track" );
#endif

							// Unlock drive locked last time
							lock( false );
						}
					}
					else
					{
#ifdef _DEBUG						
						OutputDebugString( "AUTOLOCK - 1.2 - Filename error" );
#endif

						// Unlock drive locked last time
						lock( false );
					}
				}
				break;
			
			// Stop
			case 0:
#ifdef _DEBUG
				OutputDebugString( "AUTOLOCK - 2 - Stop" );
#endif

				// Unlock drive locked last time
				lock( false );
				break;

			}
		}
		break;

	}
	return CallWindowProc( WndprocMainBackup, hwnd, message, wp, lp );
}



int init()
{
	HWND & hMain = plugin.hwndParent;

	
	// Exchange window procedure
	WndprocMainBackup = ( WNDPROC )GetWindowLong( hMain, GWL_WNDPROC );
	if( WndprocMainBackup != NULL )
	{
		SetWindowLong( hMain, GWL_WNDPROC, ( LONG )WndprocMain );
	}


	// Get Winamp.ini path
	szWinampIni = ( char * )SendMessage( hMain, WM_WA_IPC, 0, IPC_GETINIFILE );	


	// Read config
	if( szWinampIni )
	{
		int res;
		res = GetPrivateProfileInt( "gen_autolock", "Active", 1, szWinampIni );
		bActive = res ? true : false;
	}
	else
	{
		bActive = true;
	}
	
	return 0;
} 



BOOL CALLBACK WndprocConfig( HWND hwnd, UINT message, WPARAM wp, LPARAM lp )
{
	switch( message )
	{
		case WM_INITDIALOG:
		{
			// Apply settings to checkboxes
			CheckDlgButton( hwnd, IDC_LOCK, ( bActive ? BST_CHECKED : BST_UNCHECKED ) );
			
			// Set window title
			char szTitle[ 512 ] = "";
			wsprintf( szTitle, "%s %s", PLUGIN_TITLE, PLUGIN_VERSION );
			SetWindowText( hwnd, szTitle );

			// Center window on parent
			RECT rp;
			GetWindowRect( GetForegroundWindow(), &rp );
			const int iParentWidth   = rp.right - rp.left;
			const int iParentHeight  = rp.bottom - rp.top;

			RECT rf;
			GetWindowRect( hwnd, &rf );
			const int iAutoLockWidth   = rf.right - rf.left;
			const int iAutoLockHeight  = rf.bottom - rf.top;

			int ox = ( iParentWidth - iAutoLockWidth ) / 2 + rp.left;
			int oy = ( iParentHeight - iAutoLockHeight ) / 2 + rp.top;

			MoveWindow( hwnd, ox, oy, iAutoLockWidth, iAutoLockHeight, false );


			return TRUE;
		}

		case WM_SYSCOMMAND:
			switch( wp )
			{
				case SC_CLOSE:
				{
					EndDialog( hwnd, FALSE );
					return TRUE;
				}
			}
			break;

		case WM_COMMAND:
		{
			switch( LOWORD( wp ) )
			{
				case IDC_LOCK:
					bActive = ( IsDlgButtonChecked( hwnd, IDC_LOCK ) == BST_CHECKED );
					break;

				case IDOK:
					EndDialog( hwnd, FALSE );
					return TRUE;

			}
			break;

		}
	}
	return 0;
}



void config()
{
	DialogBox( plugin.hDllInstance, MAKEINTRESOURCE( IDD_CONFIG ), GetForegroundWindow(), WndprocConfig );
/*	
	MessageBox(
		NULL,
		PLUGIN_TITLE " " PLUGIN_VERSION "\n"
			"\n"
			"Copyright  2006 Sebastian Pipping  \n"
			"<webmaster@hartwork.org>\n"
			"\n"
			"-->  http://www.hartwork.org",
		"About",
		MB_ICONINFORMATION
	);
*/
} 



BOOL WritePrivateProfileInt( LPCTSTR lpAppName, LPCTSTR lpKeyName, int iValue, LPCTSTR lpFileName )
{
	TCHAR szBuffer[ 16 ];
	wsprintf( szBuffer, TEXT( "%i" ), iValue );
    return( WritePrivateProfileString( lpAppName, lpKeyName, szBuffer, lpFileName ) );
}



void quit()
{
	// Unlock drive locked last time
	lock( false );
	
	// Write config
	if( !szWinampIni ) return;
	WritePrivateProfileInt( "gen_autolock", "Active", bActive ? 1 : 0, szWinampIni );
}



extern "C" __declspec( dllexport ) winampGeneralPurposePlugin * winampGetGeneralPurposePlugin()
{
	return &plugin;
}
